#Requires -RunAsAdministrator
# Supervision Governor - Windows Installer
# Run as Administrator: powershell -ExecutionPolicy Bypass -File install-windows.ps1

param(
    [string]$RegulatorAddress = "supervision.avril-forge.com:50051",
    [string]$InstallPath = "C:\Program Files\Supervision",
    [string]$ServiceName = "SupervisionGovernor"
)

$ErrorActionPreference = "Stop"

Write-Host "=== Supervision Governor Installer ===" -ForegroundColor Cyan
Write-Host ""

# Check if running as admin
$isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
if (-not $isAdmin) {
    Write-Host "ERROR: This script must be run as Administrator" -ForegroundColor Red
    exit 1
}

# Create install directory
Write-Host "[1/5] Creating install directory..." -ForegroundColor Yellow
if (-not (Test-Path $InstallPath)) {
    New-Item -ItemType Directory -Path $InstallPath -Force | Out-Null
}

# Copy binary
Write-Host "[2/5] Installing governor binary..." -ForegroundColor Yellow
$scriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$binarySource = Join-Path $scriptDir "governor-windows.exe"
$binaryDest = Join-Path $InstallPath "governor.exe"

if (-not (Test-Path $binarySource)) {
    # Try current directory
    $binarySource = "governor-windows.exe"
}

if (-not (Test-Path $binarySource)) {
    Write-Host "ERROR: governor-windows.exe not found" -ForegroundColor Red
    exit 1
}

Copy-Item $binarySource $binaryDest -Force

# Create config file
Write-Host "[3/5] Creating configuration..." -ForegroundColor Yellow
$configPath = Join-Path $InstallPath "config.yaml"
$hostname = $env:COMPUTERNAME

$configContent = @"
# Supervision Governor Configuration
regulator_address: "$RegulatorAddress"
hostname: "$hostname"
metrics_interval: 10
heartbeat_interval: 30
log_level: info

collectors:
  - cpu
  - memory
  - disk
  - network
  - process
"@

Set-Content -Path $configPath -Value $configContent

# Stop existing service if running
Write-Host "[4/5] Configuring Windows service..." -ForegroundColor Yellow
$existingService = Get-Service -Name $ServiceName -ErrorAction SilentlyContinue
if ($existingService) {
    Write-Host "  Stopping existing service..." -ForegroundColor Gray
    Stop-Service -Name $ServiceName -Force -ErrorAction SilentlyContinue
    Start-Sleep -Seconds 2
    sc.exe delete $ServiceName | Out-Null
    Start-Sleep -Seconds 1
}

# Create Windows service using sc.exe
$binPathWithArgs = "`"$binaryDest`" --config `"$configPath`""
sc.exe create $ServiceName binPath= $binPathWithArgs start= auto displayname= "Supervision Governor" | Out-Null
sc.exe description $ServiceName "Supervision System monitoring agent - collects and reports system metrics" | Out-Null

# Configure service recovery (restart on failure)
sc.exe failure $ServiceName reset= 86400 actions= restart/5000/restart/10000/restart/30000 | Out-Null

# Set environment variables for the service
$regKey = "HKLM:\SYSTEM\CurrentControlSet\Services\$ServiceName"
$envValue = @(
    "REGULATOR_ADDRESS=$RegulatorAddress",
    "HOSTNAME=$hostname",
    "CONFIG_PATH=$configPath"
)
Set-ItemProperty -Path $regKey -Name "Environment" -Value $envValue -Type MultiString

# Start service
Write-Host "[5/5] Starting service..." -ForegroundColor Yellow
Start-Service -Name $ServiceName

# Verify
Start-Sleep -Seconds 2
$service = Get-Service -Name $ServiceName
if ($service.Status -eq "Running") {
    Write-Host ""
    Write-Host "=== Installation Complete ===" -ForegroundColor Green
    Write-Host ""
    Write-Host "Service Name:    $ServiceName"
    Write-Host "Status:          Running"
    Write-Host "Install Path:    $InstallPath"
    Write-Host "Config File:     $configPath"
    Write-Host "Regulator:       $RegulatorAddress"
    Write-Host ""
    Write-Host "Commands:" -ForegroundColor Cyan
    Write-Host "  View status:   Get-Service $ServiceName"
    Write-Host "  Stop:          Stop-Service $ServiceName"
    Write-Host "  Start:         Start-Service $ServiceName"
    Write-Host "  Uninstall:     sc.exe delete $ServiceName"
    Write-Host ""
} else {
    Write-Host ""
    Write-Host "WARNING: Service installed but not running" -ForegroundColor Yellow
    Write-Host "Check logs: Get-EventLog -LogName Application -Source $ServiceName" -ForegroundColor Yellow
}
